;;; org-autofocus.el --- Mark Forster's Auto Focus system for Org  -*- lexical-binding: t; -*-

;; Copyright (C) 2020 David Owen

;; Author: David Owen <dsowen@fugue88.ws>
;; Keywords: autofocus
;; Package-Requires: (org)

;;; Commentary:

;;; Code:

(require 'calendar)
(require 'eieio)
(require 'ewoc)

(require 'org-agenda)


(defvar org-af-marked 'org-af-marked)
(put 'org-af-marked 'org-af-marked t)
(put 'org-af-marked 'display "*")

(defun org-af--mark (point)
  "Mark the task at POINT."
  (add-text-properties point (1+ point) '(category org-af-marked)))

(defun org-af--unmark (point)
  "Unmark the task at POINT."
  (remove-text-properties point (1+ point) '(category org-af-marked)))

(defun org-af--marked-p (point)
  "Test whether the task at POINT is marked."
  (get-text-property point 'org-af-marked))


(defvar org-af-hidden 'org-af-hidden)
(put 'org-af-hidden 'org-af-hidden t)
(put 'org-af-hidden 'invisible t)


(defvar org-af-dimmed 'org-af-dimmed)
(put 'org-af-dimmed 'org-af-dimmed t)
(put 'org-af-dimmed 'face 'org-agenda-dimmed-todo-face)
(put 'org-af-dimmed 'evaporate t)

(defun org-af--dim (start end)
  "Dim the task from START to END."
  (let ((o (make-overlay start end)))
    (overlay-put o 'category org-af-dimmed)))

(defun org-af--undim (start end)
  "Undim the task from START to END."
  (remove-overlays start end 'org-af-dimmed t))

(defun org-af--dimmed-p (point)
  "Test whether the task at POINT is dimmed."
  (get-char-property point 'org-af-dimmed))


(defun org-af--beg-of-task ()
  "Move point to the beginning of the task at point.
Skips over any hidden tasks the are on the same line."
  (beginning-of-line)
  (when (get-text-property (point) 'org-af-hidden)
    (goto-char (next-single-property-change (point) 'org-af-hidden))))


(defun org-af-mark ()
  "Mark or unmark the current task, dim the previous marked task if any, and hide any tasks between."
  (interactive)
  (org-af--beg-of-task)

  (let ((inhibit-read-only t)
        (task-point (point))
        bol marked)

    (unless (org-af--dimmed-p task-point)
      (setq marked (not (org-af--marked-p task-point)))
      (if marked
          (org-af--mark task-point)
        (org-af--unmark task-point))

      (save-excursion
        ;; Go to previous mark, if any
        (while (and (= (forward-line -1) 0)
                    (setq bol (line-beginning-position))
                    (get-text-property bol 'type)
                    (not (org-af--marked-p bol))))

        (when (org-af--marked-p bol)
          (if marked
              (org-af--dim bol (line-end-position))
            (org-af--undim bol (line-end-position))))

        (forward-line)
        (setq bol (line-beginning-position))
        (if marked
            (add-text-properties bol task-point '(category org-af-hidden))
          (remove-text-properties bol task-point '(category org-af-hidden)))))))


(defun org-af--timestamp ()
  "Return the current local time, with nanoseconds, formatted as an ISO string."
  (format-time-string "%Y-%m-%d %H:%M:%S.%N" (current-time)))


(defun org-af-defer (&optional no-redisplay)
  "Defer the current task and update the display, unless NO-REDISPLAY."
  (interactive)
  (org-af--beg-of-task)
  (if (org-af--dimmed-p (point))
      (message "dimmed; skipping")
    (when (org-af--marked-p (point))
      (org-af-mark))

    ;; This comes from the guts of the org-agenda-set-property command.
    (let* ((hdmarker (or (org-get-at-bol 'org-hd-marker)
		         (org-agenda-error)))
	   (buffer (marker-buffer hdmarker))
	   (pos (marker-position hdmarker))
	   (inhibit-read-only t)
           (time (org-af--timestamp)))
      (org-with-remote-undo buffer
        (with-current-buffer buffer
	  (widen)
	  (goto-char pos)
	  (org-show-context 'agenda)
	  (org-set-property "TOUCHED" time))))

    (save-excursion
      (let* ((inhibit-read-only t)
             (text (delete-and-extract-region (point)
                                              (1+ (line-end-position)))))
        (goto-char (point-max))
        (save-excursion
          (insert text))))))


(defun org-af-complete ()
  (interactive)
  (org-af--beg-of-task)
  (when (org-af--marked-p (point))
    (org-af-mark))
  (org-agenda-todo)
  (org-af-defer t))


(defvar org-autofocus-mode-map (make-sparse-keymap))
(define-key org-autofocus-mode-map "." 'org-af-mark)
(define-key org-autofocus-mode-map "d" 'org-af-defer)
(define-key org-autofocus-mode-map "c" 'org-af-complete)
(define-key org-autofocus-mode-map "t" 'org-af-complete)


(defun org-af--touched (entry)
  (let* ((marker (get-text-property 0 'org-hd-marker entry))
         (touched (org-entry-get marker "TOUCHED")))
    (unless touched
      (setq touched (org-af--timestamp))
      (org-entry-put marker "TOUCHED" touched))
    touched))


(defun org-af--cmp (a b)
  "Compare A to B.
If A > B, return +1; if A < B, return -1; else, return NIL."
  (let* ((a-touched (org-af--touched a))
         (b-touched (org-af--touched b))
         (c (compare-strings a-touched nil nil b-touched nil nil)))
    (cond
     ((eq c t)  nil)
     ((< c 0)   -1)
     ((> c 0)   +1)
     (t (error "%S and %S compared as %S" a-touched b-touched c)))))


(define-minor-mode org-autofocus-mode nil nil " AF" org-autofocus-mode-map
  ;;(setq org-agenda-redo-command '(org-autofocus))
  )


(defun org-autofocus ()
  (interactive)
  (let ((org-agenda-cmp-user-defined 'org-af--cmp)
        (org-agenda-sorting-strategy '(user-defined-up)))
    (org-agenda nil "n")
    (org-autofocus-mode)))


(add-hook 'org-agenda-mode-hook 'org-autofocus-mode)
(remove-hook 'org-agenda-mode-hook 'org-autofocus-mode)


;; (defvar-local org-af--ewoc nil)


;; (defclass org-af--task ()
;;   ((org :initarg :org)
;;    (marked :initform nil)
;;    (passed :initform nil)))


;; (defun org-af--pp (obj)
;;   "Print the task OBJ in the current buffer at point."
;;   (with-slots (org marked passed) obj
;;     (when (or marked (not passed))
;;       (insert (if marked "*" " ")
;;               " "
;;               (if passed
;;                   (propertize org 'face 'org-agenda-dimmed-todo-face)
;;                 org)
;;               "\n"))))


;; (define-derived-mode org-autofocus-mode special-mode "Org AF" nil
;;   (let ((inhibit-read-only t)
;;         (date (calendar-current-date))
;;         tasks)
;;     (setq buffer-undo-list t)

;;     (dolist (f (org-agenda-files))
;;       (catch 'nextfile
;; 	(org-check-agenda-file f)
;; 	(setq tasks (nconc (org-agenda-get-day-entries f date :todo) tasks))))

;;     (setq org-af--ewoc (ewoc-create 'org-af--pp "header\n" "footer\n" t))
;;     (dolist (task tasks)
;;       (ewoc-enter-last org-af--ewoc (make-instance 'org-af--task :org task)))))


;; (defun org-af--mark ()
;;   "Toggle whether the task at point is marked."
;;   (interactive)
;;   (let* ((node (ewoc-locate org-af--ewoc))
;;          (obj (ewoc-data node))
;;          changed)
;;     (unless (slot-value obj 'passed)
;;       (let ((marked (not (slot-value obj 'marked))))
;;         (setf (slot-value obj 'marked) marked)
;;         (push node changed)
;;         (let ((node node))
;;           (while (and (setf node (ewoc-prev org-af--ewoc node))
;;                       (setf obj (ewoc-data node))
;;                       (not (slot-value obj 'marked)))
;;             (setf (slot-value obj 'passed) marked)
;;             (push node changed))
;;           (when node
;;             (setf (slot-value obj 'passed) marked)
;;             (push node changed))))
;;       (apply 'ewoc-invalidate org-af--ewoc changed)
;;       (ewoc-goto-node org-af--ewoc node))))


;; (defun org-af--complete ()
;;   "Complete the task at point, if it's marked and not passed over."
;;   (interactive)
;;   (let* ((node (ewoc-locate org-af--ewoc))
;;          (obj (ewoc-data node)))
;;     (unless (and nil (or (slot-value obj 'passed)
;;                 (not (slot-value obj 'marked))))
;;       (org-agenda-todo)
;;       (org-autofocus-mode))))


;; (define-key org-autofocus-mode-map "." 'org-af--mark)
;; (define-key org-autofocus-mode-map "c" 'org-af--complete)


;; ;;;###autoload
;; (defun org-autofocus ()
;;   "Show the AutoFocus buffer; create it if necessary."
;;   (interactive)
;;   (pop-to-buffer (get-buffer-create "*Org AutoFocus*"))
;;   (org-autofocus-mode))


(provide 'org-autofocus)
;;; org-autofocus.el ends here
