#!/usr/bin/python3

from random import shuffle

from player import Player
import ai


def spies_for_players(player_count):
    return {
        5: 2,
        6: 2,
        7: 3,
        8: 3,
        9: 3,
        10: 4
    }[player_count]


def mission_size(player_count, mission_num):
    return {
        5: [2, 3, 2, 3, 3],
        6: [2, 3, 4, 3, 4],
        7: [2, 3, 3, 4, 4],
        8: [3, 4, 4, 5, 5],
        9: [3, 4, 4, 5, 5],
        10: [3, 4, 4, 5, 5]
    }[player_count][mission_num - 1]


def required_failures(player_count, mission_num):
    if mission_num == 4 and player_count >= 7:
        return 2
    else:
        return 1


class Shared(object):

    def __init__(self, player_count, players, role_deck):
        self.player_count = player_count
        self.players = players
        self.role_deck = role_deck
        self.mission = 0
        self.captain = 0
        self.team_failures = 0
        self.wins = 0
        self.losses = 0

    def current_mission_size(self):
        team_size = mission_size(self.player_count, self.mission)
        print("Mission {} needs {} members".format(self.mission, team_size))
        return team_size

    def next_mission(self):
        self.mission += 1

    def get_proposed_team(self, team_size):
        print("Captain is {}".format(self.captain))
        team = self.players[self.captain].propose_team(team_size)
        print("He proposed {}".format(team))
        return team

    def next_team_captain(self):
        self.captain = (self.captain + 1) % self.player_count

    def reset_team_failures(self):
        self.team_failures = 0

    def record_team_failure(self):
        self.team_failures += 1

    def get_mission_failures(self, team):
        failures = 0
        for i in team:
            if self.role_deck[i] == "SPY":
                if not self.players[i].perform_mission():
                    failures = failures + 1
        return failures

    def required_failures(self):
        return required_failures(self.player_count, self.mission)

    def mission_succeeded(self):
        print("Mission succeeded")
        self.wins += 1

    def mission_failed(self, failures):
        print("Mission had {} failures".format(failures))
        self.losses += 1

    def game_over(self):
        if self.team_failures == 5:
            print("The spies won!")
            for p in self.players:
                p.observe_game(False)
            return True
        elif self.wins >= 3:
            print("The Resistance won!")
            for p in self.players:
                p.observe_game(True)
            return True
        elif self.losses >= 3:
            print("The Spies won!")
            for p in self.players:
                p.observe_game(False)
            return True
        else:
            return False


class AdvanceMission(object):

    def __init__(self, shared):
        self.shared = shared

    def do(self):
        self.shared.next_mission()
        team_size = self.shared.current_mission_size()
        return AssembleTeam(self.shared, team_size)


class AssembleTeam(object):

    def __init__(self, shared, team_size):
        self.shared = shared
        self.team_size = team_size

    def do(self):
        team = self.shared.get_proposed_team(self.team_size)

        votes = [p.approve_team(team) for p in self.shared.players]
        approvals = sum(votes)
        approved = approvals > self.shared.player_count / 2

        print("Team {}: {} for, {} against".format(
            {True: "approved", False: "rejected"}[approved],
            approvals, self.shared.player_count - approvals
        ))

        for p in self.shared.players:
            p.observe_team_vote(approved, votes)

        self.shared.next_team_captain()

        if approved:
            self.shared.reset_team_failures()
            return PerformMission(self.shared, team)
        else:
            self.shared.record_team_failure()
            return AssembleTeam(self.shared, self.team_size)


class PerformMission(object):

    def __init__(self, shared, team):
        self.shared = shared
        self.team = team

    def do(self):
        failures = self.shared.get_mission_failures(self.team)
        success = failures < self.shared.required_failures()

        for p in self.shared.players:
            p.observe_mission(success, failures)

        if success:
            self.shared.mission_succeeded()
        else:
            self.shared.mission_failed(failures)

        return AdvanceMission(self.shared)


def main(player_count):
    spy_count = spies_for_players(player_count)
    resistance_count = player_count - spy_count
    print("Will have {} spies and {} resistance".format(spy_count, resistance_count))
    role_deck = ["RES"] * resistance_count + ["SPY"] * spy_count
    shuffle(role_deck)
    print(role_deck)
    players = [Player(player_count, i, r) if r == "SPY"
               else ai.Player(player_count, i, r)
               for i, r in enumerate(role_deck)]

    spy_nums = [i for i, r in enumerate(role_deck) if r == "SPY"]

    for i, r in enumerate(role_deck):
        if r == "SPY":
            players[i].reveal_spies(spy_nums)

    shared = Shared(player_count, players, role_deck)
    state = AdvanceMission(shared)

    while not shared.game_over():
        state = state.do()


if __name__ == '__main__':
    main(10)
